﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.Common;

namespace VA.PPMS.IWS.MappingService.Mappers
{
    public abstract class MapperBase
    {
        public Guid NetworkId { get; set; }

        protected bool IsWithinContext { get; set; }

        protected readonly IPpmsContextHelper PpmsContextHelper;
        protected readonly IPpmsHelper PpmsHelper;

        private PpmsContext _context;
        protected PpmsContext Context
        {
            get
            {
                if (_context == null) _context = PpmsContextHelper.GetContextAsync().GetAwaiter().GetResult();
                return _context;
            }
            set
            {
                _context = value;
                // If context is set, use provided context for update activities
                IsWithinContext = _context != null;
            }
        }

        protected MapperBase(IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper)
        {
            PpmsContextHelper = ppmsContextHelper;
            PpmsHelper = ppmsHelper;
        }
        
        public IEnumerable<SetStateRequest> EntityDelete<T>(IEnumerable<T> list)
        {
            var requests = new List<SetStateRequest>();
            var items = list as T[] ?? list.ToArray();
            if (!items.Any()) return requests;

            requests.AddRange(items.Select(EntityDelete).Where(request => request != null));

            return requests;
        }

        public SetStateRequest EntityDelete<T>(T entity)
        {
            return entity != null ? DeactivateEntity(entity) : null;
        }

        protected SetStateRequest DeactivateEntity<T>(T entity)
        {
            return PpmsHelper.SetEntityStatus(entity, (int)AccountState.Inactive, (int)Account_StatusCode.Inactive);
        }

        protected SetStateRequest ActivateEntity<T>(T entity)
        {
            return PpmsHelper.SetEntityStatus(entity, (int)AccountState.Active, (int)Account_StatusCode.Active);
        }

        protected bool IsChanged(string source, string target)
        {
            return !string.IsNullOrEmpty(source) && source != target;
        }

        protected void HandleException(string message)
        {
            throw new PpmsServiceException(message);
        }

        protected void HandleException(string message, string id)
        {
            HandleException($"{message} [Entity Id: {id}]");
        }

        protected IEnumerable<T> ConvertEntityList<T>(IEnumerable<object> entityList)
        {
            var objects = entityList as object[] ?? entityList.ToArray();
            if (!objects.Any()) return new List<T>();
            var list = objects.ToList();

            return list.ConvertAll(x => (T)x);
        }

        protected void SetNetworkByShorthand(string shorthand)
        {
            if (string.IsNullOrEmpty(shorthand)) return;

            var id = PpmsHelper.GetNetworkIdByShorthand(shorthand).GetAwaiter().GetResult();

            if (id.HasValue)
            {
                NetworkId = id.Value;
            }
        }

        protected async Task<Entity> GetProviderByCorrelationId(string correlationId, bool loadProperties = false)
        {
            using (var context = await PpmsContextHelper.GetContextAsync())
            {
                var id = new Guid(correlationId);
                var entity = context.AccountSet.FirstOrDefault(a => a.AccountId == id);
                if (loadProperties) LoadProviderProperties(entity, context);

                return entity;
            }
        }

        protected async Task<Entity> GetProviderByNpi(string npi, bool loadProperties = false)
        {
            using (var context = await PpmsContextHelper.GetContextAsync())
            {
                var entity = context.ppms_provideridentifierSet.FirstOrDefault(n => n.ppms_ProviderIdentifier == npi);
                if (entity == null) return null;

                context.LoadProperty(entity, "ppms_account_ppms_provideridentifier_Provider");

                if (entity.ppms_account_ppms_provideridentifier_Provider == null) return null;

                var provider = entity.ppms_account_ppms_provideridentifier_Provider;
                if (loadProperties) LoadProviderProperties(provider, context);

                return provider;
            }
        }

        protected async Task<va_state> GetState(string state)
        {
            using (var context = await PpmsContextHelper.GetContextAsync())
            {
                if (context != null)
                {
                    return context.va_stateSet.FirstOrDefault(s => s.va_name == state);
                }

                HandleException("GetState: Unable to retrieve context.");
                return null;
            }
        }

        protected async Task<ppms_taxonomy> GetTaxonomy(string specialty)
        {
            using (var context = await PpmsContextHelper.GetContextAsync())
            {
                if (context != null)
                {
                    return context.ppms_taxonomySet.FirstOrDefault(s => s.ppms_SpecialtyCode == specialty);
                }

                HandleException("GetState: Unable to retrieve context.");
                return null;
            }
        }

        protected void LoadProviderProperties(Entity entity, PpmsContext context)
        {
            if (entity == null || context == null) return;

            context.LoadProperty(entity, "ppms_account_ppms_provideridentifier_Provider");
            context.LoadProperty(entity, "ppms_account_ppms_providerservice");
            context.LoadProperty(entity, "ppms_account_ppms_boardcertification");
            context.LoadProperty(entity, "ppms_account_organizationauthorizedofficial");
            context.LoadProperty(entity, "ppms_account_ppms_otherprovideridentifier");
            context.LoadProperty(entity, "contact_customer_accounts");
            context.LoadProperty(entity, "ppms_account_providerlicensure");
            context.LoadProperty(entity, "ppms_account_ppms_othername");
            context.LoadProperty(entity, "ppms_account_ppms_providertaxonomy");
            context.LoadProperty(entity, "ppms_account_deascheduleprivilege");
            context.LoadProperty(entity, "ppms_account_providernetworkid");
        }
    }
}